#  ---------------------------------------------------------------------
#   General database implementation of ODL-Browse (v1.0) - service prov.  
#    v1.1
#    April 2002
#  ------------------+--------------------+-----------------------------
#   Hussein Suleman  |   hussein@vt.edu   |    www.husseinsspace.com    
#  ------------------+--------------------+-+---------------------------
#   Department of Computer Science          |        www.cs.vt.edu       
#     Digital Library Research Laboratory   |       www.dlib.vt.edu      
#  -----------------------------------------+-------------+-------------
#   Virginia Polytechnic Institute and State University   |  www.vt.edu  
#  -------------------------------------------------------+-------------


package ODL::DBBrowse::BrowseSP;


use Pure::Log;
use Pure::X2D;
use Pure::EZXML;
use ODL::DBBrowse::Browse;

use XOAI::Harvester;
use vars ('@ISA');
@ISA = ("XOAI::Harvester");


# constructor
sub new
{
   my ($classname, $configfile) = @_;

   my $self = $classname->SUPER::new ();

   # get configuration from file
   my $con = new Pure::X2D ($configfile);
   $self->{'database'} = $con->param ('database', '');
   $self->{'dbusername'} = $con->param ('dbusername', '');
   $self->{'dbpassword'} = $con->param ('dbpassword', '');
   $self->{'dbindex'} = $con->param ('table', 'dbbrowse');
   $self->{'xmlroot'} = 'metadata';
   $self->{'log'} = new Pure::Log ('harvest.log');
   $self->{'counter'} = 0;
   $self->{'archive'} = $con->{'archive'};
   
   # get timestamp
   my $timestamp = time;
   if (-e $configfile)
   {
      $timestamp = (stat ($configfile))[9];
   }

   # create browse engine object
   $self->{'browse'} = new ODL::DBBrowse::Browse 
     ($self->{'database'}, $self->{'dbusername'}, 
      $self->{'dbpassword'}, $self->{'dbindex'},
      $self->{'xmlroot'}, $timestamp, $con->{'browser'});
   
   bless $self, $classname;
   return $self;
}


# destructor
sub dispose
{
   my ($self) = @_;
   $self->{'browse'}->dispose;
}


# pass parameters and archive info to harvester
sub harvest
{
   my ($self, $when, $what, $fromwhere) = @_;
   $self->SUPER::harvest ($self->{'archive'}, $when, $what, $fromwhere);   
}


# output log file header
sub processHarvestStart
{
   my ($self, $archiveidentifier, $archiveurl, $archiveinterval, 
       $archivemetadataPrefixes, $archiveset, $archivedaysoverlap,
       $archiveinterrequestgap) = @_;
   $self->{'log'}->log ("");
   $self->{'log'}->log ("Harvesting:");
   $self->{'log'}->log ("  archive         = $archiveidentifier");
   $self->{'log'}->log ("  url             = $archiveurl");
   $self->{'log'}->log ("  interval        = $archiveinterval days");
   foreach my $archivemetadataPrefix (@$archivemetadataPrefixes)
   {
      $self->{'log'}->log ("  metadataPrefix  = $archivemetadataPrefix");
   }
   $self->{'log'}->log ("  set             = $archiveset");
   $self->{'log'}->log ("  daysoverlap     = $archivedaysoverlap");
   $self->{'log'}->log ("  interrequestgap = $archiveinterrequestgap");
   $self->{'log'}->log ("");
}


# close down engine
sub processHarvestEnd
{
   my ($self) = @_;
   $self->{'log'}->log ("Done harvesting.");
   $self->{'log'}->log ("");
}


# remove old traces of an identifier from engine
sub processIdStart
{
   my ($self, $identifier, $status) = @_;
   
   $self->{'counter'}++;
   $self->{'log'}->log ("[$self->{counter}] Processing : $identifier");
   $self->{'browse'}->unindexIdentifier ($identifier);
}


# index relevant fields in a record
sub processRecord
{
   my ($self, $xml, $date, $metadataPrefix, $set, $archiveid) = @_;

   # get identifier from header of record
   my $identifier = '';
   if ($xml->getElementsByTagName ('header', 0)->getLength == 1)
   {
      my $header = $xml->getElementsByTagName ('header', 0)->item(0);
      if ($header->getElementsByTagName ('identifier', 0)->getLength == 1)
      {
         my $identifiercontainer = $header->getElementsByTagName ('identifier', 0)->item(0);
         if (($identifiercontainer->hasChildNodes) && 
             ($identifiercontainer->getFirstChild->getNodeType == TEXT_NODE))
         {
            $identifier = $identifiercontainer->getFirstChild->getNodeValue;
         }
      }
   }
   $identifier =~ s/^([\s\r\n\t]*)([^\s\r\n\t]*)([\s\r\n\t]*)$/$2/;

   # get metadata pointer from record
   my $metadata = '';
   if ($xml->getElementsByTagName ('metadata', 0)->getLength == 1)
   {
      $metadata = $xml->getElementsByTagName ('metadata', 0)->item(0);
   }
   
   my $status = $xml->getAttribute ('status');
   
   if ($status eq 'deleted')
   {
      $self->{'log'}->log ("[$self->{counter}] Deleting: $identifier");
      $self->{'browse'}->unindexIdentifier ($identifier);
   }
   elsif (($identifier ne '') && ($metadata ne ''))
   {
      $self->{'log'}->log ("[$self->{counter}] Indexing: $identifier, $metadataPrefix");
      $self->{'browse'}->indexXML ($metadata, $identifier);
   }
}


# commit browse engine buffers
sub processIdEnd
{
   my ($self, $identifier) = @_;
   
   $self->{'log'}->log ("[$self->{counter}] Done with : $identifier");
}


1;

