#  ---------------------------------------------------------------------
#   General database implementation of ODL-Browse (v1.0) - data provider
#    v1.1
#    April 2002
#  ------------------+--------------------+-----------------------------
#   Hussein Suleman  |   hussein@vt.edu   |    www.husseinsspace.com    
#  ------------------+--------------------+-+---------------------------
#   Department of Computer Science          |        www.cs.vt.edu       
#     Digital Library Research Laboratory   |       www.dlib.vt.edu      
#  -----------------------------------------+-------------+-------------
#   Virginia Polytechnic Institute and State University   |  www.vt.edu  
#  -------------------------------------------------------+-------------


package ODL::DBBrowse::BrowseDP;


use Pure::EZXML;
use Pure::X2D;
use XOAI::XOAISP;
use ODL::DBBrowse::Browse;

use XOAI::XOAIDP;
use vars ('@ISA');
@ISA = ("XOAI::XOAIDP");


# constructor
sub new
{
   my ($classname, $configfile) = @_;
   
   my $self = $classname->SUPER::new ();

   # get configuration from file
   my $con = new Pure::X2D ($configfile);
   $self->{'database'} = $con->param ('database', '');
   $self->{'dbusername'} = $con->param ('dbusername', '');
   $self->{'dbpassword'} = $con->param ('dbpassword', '');
   $self->{'dbindex'} = $con->param ('table', 'irdb');
   $self->{'xmlroot'} = 'metadata';
   $self->{'repositoryName'} = $con->param ('repositoryName', 'ODL Search Engine');
   $self->{'adminEmail'} = $con->param ('adminEmail', "mailto:someone\@somewhere");
   $self->{'sourcearchive'} = $con->param ('archive/url', '');

   # get timestamp
   my $timestamp = time;
   if (-e $configfile)
   {
      $timestamp = (stat ($configfile))[9];
   }

   # create browse engine object
   $self->{'browse'} = new ODL::DBBrowse::Browse 
     ($self->{'database'}, $self->{'dbusername'}, 
      $self->{'dbpassword'}, $self->{'dbindex'},
      $self->{'xmlroot'}, $timestamp, $con->{'browser'});

   $self->{'odlsearch_xmlnsprefix'} = 'http://oai.dlib.vt.edu/ODL/Search/1.0',

   bless $self, $classname;
   return $self;
}


# destructor
sub dispose
{
   my ($self) = @_;
   $self->SUPER::dispose ();
   $self->{'browse'}->dispose;
}


# issue a query to the engine
sub browse
{
   my ($self, $query, $start, $stop) = @_;
   return $self->{'browse'}->browse ($query, $start, $stop);
}


# component-specific identification
sub Archive_Identify
{
   my ($self) = @_;
   
   my $id = $self->SUPER::Archive_Identify ();
      
   if (! exists $id->{'description'})
   {
      $id->{'description'} = [];
   }
   my $desc = {
      'odl-description' => [[ 
         {
            'xmlns' => $self->{xoai_xmlnsprefix}.'/odl-description',
            'xsi:schemaLocation' => 
                       $self->{xoai_xmlnsprefix}.'/odl-description '.
                       $self->{xoai_xmlnsprefix}.'/odl-description.xsd'
         },
         {
            'protocol' => 'ODL-Browse',
            'version'  => '1.0',
            'mdorder'  => [ qw ( protocol version ) ]
         } 
      ]]
   };
   push (@{$id->{'description'}}, $desc);
   
   $id;
}


# retrieve records from the source archive as required
sub Archive_FormatRecord
{
   my ($self, $hashref, $metadataFormat) = @_;
   
   my $oai = new XOAI::XOAISP ($self->{sourcearchive});
   
   my $output = $oai->doXOAI ('GetRecord', 
                              { metadataPrefix => $metadataFormat,
                                identifier => $hashref->{urn} }
                             ) -> content;
   
   while (substr ($output, 0, 8) ne '<record>')
   { $output = substr ($output, 1); }
   
   while (substr ($output, -9) ne '</record>')
   { $output = substr ($output, 0, -1); }

   $output;
}


# get list of sets from archive
sub Archive_ListSets
{
   my ($self) = @_;
   
   my $browsesets = $self->{'browse'}->getsets;
   
   my @xoaisets = map { [ $_->[0].'('.$_->[1].')', $_->[0].'('.$_->[1].')' ] } @$browsesets;
   
   \@xoaisets;
}

# ignore verb
sub Archive_ListMetadataFormats
{
   my ($self, $identifier) = @_;
   
   [];
}


# pass request through to source
sub Archive_GetRecord
{
   my ($self, $identifier) = @_;
   
   my %records = ( urn => $identifier );

   \%records;
}


# perform browse operation and return list of identifiers
sub Archive_ListRecords
{
   my ($self, $set, $from, $until, $metadataPrefix, $resumptionToken) = @_;
   
   my $results = [ 0 ];
   my @allrows = ();
   
   if ($set =~ /odlbrowse1\/(.*)/)
   { 
      # separate into components by removing last two fields
      my $query = $1;
      my $lastslash = rindex ($query, '/');
      if ($lastslash > -1)
      {
         my $stop = int(substr ($query, $lastslash+1));
         $query = substr ($query, 0, $lastslash);
         my $secondlastslash = rindex ($query, '/');
         if ($secondlastslash > -1)
         {
            my $start = int(substr ($query, $secondlastslash+1));
            $query = substr ($query, 0, $secondlastslash);
 
            # issue query to engine
            $query =~ s/_/ /g;
            $results = $self->browse ($query, $start, $stop);
         }
      }
   }
   
   for ( my $i = 0; $i<$#$results; $i++ )
   {
      push (@allrows, { urn => $results->[$i] });
   }
   $resumptionToken = '';
   my $responseContainer =
      "<hits ".
      "xmlns=\"$self->{'odlsearch_xmlnsprefix'}/responsecontainer\" ".
      "xsi:schemaLocation=\"$self->{'odlsearch_xmlnsprefix'}/responsecontainer ".
      "$self->{'odlsearch_xmlnsprefix'}/responsecontainer.xsd\">".
      $results->[$#$results].
      "</hits>";

   ( \@allrows, $resumptionToken, $responseContainer, $metadataPrefix );
}


# forward request to ListRecords handler
sub Archive_ListIdentifiers
{
   my ($self, $set, $from, $until, $resumptionToken) = @_;

   $self->Archive_ListRecords ($set, $from, $until, '', $resumptionToken);
}


1;

