#!/usr/bin/perl -w -I.
use strict;

#  ---------------------------------------------------------------------
#   Browsing Engine
#    v1.1
#    April 2002
#  ------------------+--------------------+-----------------------------
#   Hussein Suleman  |   hussein@vt.edu   |    www.husseinsspace.com    
#  ------------------+--------------------+-+---------------------------
#   Department of Computer Science          |        www.cs.vt.edu       
#     Digital Library Research Laboratory   |       www.dlib.vt.edu      
#  -----------------------------------------+-------------+-------------
#   Virginia Polytechnic Institute and State University   |  www.vt.edu  
#  -------------------------------------------------------+-------------


package ODL::DBBrowse::Browse;


use Pure::EZXML;
use DBI;


# constructor
sub new
{
   my ($classname, $database, $dbusername, $dbpassword, $dbindex, $rootxml, $timestamp, $browser) = @_;
   
   # cater for missing parameters
   if (! defined $classname)
   { die "This method has to be called as a constructor"; }
   if (! defined $database)
   { die "Missing database name"; }
   if (! defined $dbusername)
   { $dbusername = ''; }
   if (! defined $dbpassword)
   { $dbpassword = ''; }
   if (! defined $dbindex)
   { $dbindex = 'brindex'; }
   if (! defined $rootxml)
   { $rootxml = 'metadata'; }
   
   my $self = {
      database        => $database,
      dbusername      => $dbusername,
      dbpassword      => $dbpassword,
      dbindex         => $dbindex,
      
      rootxml         => $rootxml,

      browser         => $browser,
   };
   bless $self, $classname;
   
   $self->openDatabase ($timestamp);
   
   return $self;
}


# open the database, create the tables if necessary (and optionally delete the
# old tables if the configuration has changed since they were last created)
sub openDatabase
{
   my ($self, $timestamp) = @_;

   $self->{'dbh'} = DBI->connect ($self->{'database'}, $self->{'dbusername'}, $self->{'dbpassword'}, { PrintError => 0 });
   
   # check for correspondence between table and configuration
   my $sth = $self->{'dbh'}->prepare ("select value from $self->{'dbindex'}admin where name=\'date\'");
   $sth->execute;
   my @row = $sth->fetchrow_array;
   $sth->finish;
   if ($#row > -1)
   {
      if ($row[0] > $timestamp)
      { return; }
   }
   
   # remove old tables from database
   $sth = $self->{'dbh'}->prepare ("select value from $self->{'dbindex'}admin where name=\'table\'");
   $sth->execute;
   while (my $table = $sth->fetchrow_array)
   {
      $self->{'dbh'}->do ("drop table $table");
   }
   $sth->finish;
   $sth = $self->{'dbh'}->do ("drop table $self->{'dbindex'}");
   
   # make individual tables in database
   my $rows = '';
   foreach my $browser (@{$self->{'browser'}})
   {
      my $name = $browser->{'name'}->[0];
      
      $sth = $self->{'dbh'}->do ("create table $self->{'dbindex'}$name (".
                                 "identifier char(100) not null, ".
                                 "$name char(100) not null".
                                 ")");
      $sth = $self->{'dbh'}->do ("create index $name on $self->{'dbindex'}$name ($name)");
      $sth = $self->{'dbh'}->do ("create index identifier on $self->{'dbindex'}$name (identifier)");
                                      
      $rows .= "$name char(100) not null, ";
   }
   
   # make master sorting table
   $sth = $self->{'dbh'}->do ("create table $self->{'dbindex'} (".
                              "identifier char(100) not null, ".
                              $rows.
                              "primary key (identifier)".
                              ")");

   # create indices for master table
   foreach my $browser (@{$self->{'browser'}})
   {
      my $name = $browser->{'name'}->[0];
      $sth = $self->{'dbh'}->do ("create index $name on $self->{'dbindex'} ($name)");
   }
   
   # create admin table and update with current status
   my $newtime = time;
   $sth = $self->{'dbh'}->do ("create table $self->{'dbindex'}admin (".
                              "name char(50) not null, ".
                              "value char(100) not null".
                              ")");
   $sth = $self->{'dbh'}->do ("delete from $self->{'dbindex'}admin where name=\'date\'");
   $sth = $self->{'dbh'}->do ("insert into $self->{'dbindex'}admin values (\'date\', \'$newtime\')");
   $sth = $self->{'dbh'}->do ("delete from $self->{'dbindex'}admin where name=\'table\'");
   foreach my $browser (@{$self->{'browser'}})
   {
      my $name = $browser->{'name'}->[0];
      $sth = $self->{'dbh'}->do ("insert into $self->{'dbindex'}admin values (\'table\', \'$self->{'dbindex'}$name\')");
   }
}


# close the database
sub dispose
{
   my ($self) = @_;
   
   $self->{'dbh'}->disconnect;
}


# destructor
sub DESTROY
{
   my ($self)= @_;
   
   $self->dispose;
}


# get a listing of all the single parameter sets to sort or categorize on
sub getsets
{
   my ($self) = @_;

   my @resultsbrowse = ();
   my @resultssort = ();
   foreach my $browser (@{$self->{'browser'}})
   {
      my $name = $browser->{'name'}->[0];
      my $type = $browser->{'type'}->[0];
      if ($type eq 'controlled')
      {
         my $sth = $self->{'dbh'}->prepare ("select distinct $name from $self->{'dbindex'}$name");
         $sth->execute;
         while (my @row = $sth->fetchrow_array)
         {
            push (@resultsbrowse, [$name, $row[0]]);
         }
         $sth->finish;
         push (@resultssort, ['sort', $name]);
      }
      else
      {
         push (@resultssort, ['sort', $name]);
      }
   }
   
   my @results = (@resultsbrowse, @resultssort);
   
   \@results;   
}


# return a list of identifiers that correspond to the given query
sub browse
{
   my ($self, $query, $start, $stop) = @_;

   # split query into separate parts based on parentheses
   my @querypart = split (/[\(\)]/, $query);
   
   # check for wrong number of or empty parameters
   if ((@querypart % 2) == 1)
   {
      return [];
   }
   
   # create sql query
   my @sqlwheres = ();
   my @sqlsorts = ();
   my @tables = ( $self->{'dbindex'} );
   my $sqlquery = '';
   for ( my $i=0; $i<=$#querypart; $i+=2 )
   {
      if ($querypart[$i] eq 'sort')
      { 
#         my @sortpart = split (',', $querypart[$i+1]);
         my @sortpart = ($querypart[$i+1]);
         foreach my $j (@sortpart)
         {
            if (substr ($j, -1) eq '+')
            {
               $j = substr ($j, 0, -1);
               push (@sqlsorts, "$self->{'dbindex'}.$j asc");
            }
            elsif (substr ($j, -1) eq '-')
            {
               $j = substr ($j, 0, -1);
               push (@sqlsorts, "$self->{'dbindex'}.$j desc");
            }
            else
            {
               push (@sqlsorts, "$self->{'dbindex'}.$j");
            }
         }
      }
      else
      {
#         my @catpart = split (',', $querypart[$i+1]);
         my @catpart = ($querypart[$i+1]);
         my @sqlors = ();
         foreach my $j (@catpart)
         {
            push (@sqlors, "$self->{'dbindex'}$querypart[$i].$querypart[$i]=\'$j\'");
         }
         my $orexpr = join (' or ', @sqlors);
         push (@sqlwheres, '('.$orexpr.')');
         push (@sqlwheres, "$self->{'dbindex'}$querypart[$i].identifier=$self->{'dbindex'}.identifier");
         push (@tables, "$self->{'dbindex'}$querypart[$i]");
      }
   }
   my $listoftables = join (' join ', @tables);
   $sqlquery = "select distinct $self->{'dbindex'}.identifier from $listoftables";

   if ($#sqlwheres >= 0)
   {
      $sqlquery .= ' where '.join (' and ', @sqlwheres);
   }
   if ($#sqlsorts >= 0)
   {
      $sqlquery .= ' order by '.join (',', @sqlsorts);
   }
   else
   {
      $sqlquery .= " group by $self->{'dbindex'}.identifier";
   }

   if ((defined $start) && ($start ne '') && ($start > 0))
   {
      if ((defined $stop) && ($stop ne '') && ($stop > 0))
      { $sqlquery .= ' limit '.($start-1).','.($stop-$start+1); }
      else
      { $sqlquery .= ' limit '.($start-1).',-1'; }
   }
   else
   {
      if ((defined $stop) && ($stop ne '') && ($stop > 0))
      { $sqlquery .= ' limit '.($stop+1); }
   }
      
   my @results = ();
   
#print $sqlquery;   
   
   # issue sql statement and get results
   my $sth = $self->{'dbh'}->prepare ($sqlquery);
   $sth->execute;
   while (my @row = $sth->fetchrow_array)
   {
      push (@results, $row[0]);
   }
   $sth->finish;
   
   # get complete list size and append it to the list of results
   $sqlquery = "select count(distinct $self->{'dbindex'}.identifier) from $listoftables";
   if ($#sqlwheres >= 0)
   {
      $sqlquery .= ' where '.join (' and ', @sqlwheres);
   }
   $sth = $self->{'dbh'}->prepare ($sqlquery);
   $sth->execute;
   while (my @row = $sth->fetchrow_array)
   {
      push (@results, $row[0]);
   }
   $sth->finish;
   
   \@results;
}


# remove all previous index entries for the given identifier
sub unindexIdentifier
{
   my ($self, $identifier) = @_;
   
   # remove all entries from the database
   $self->{'dbh'}->do ("delete from $self->{'dbindex'} where identifier=\'$identifier\'");
   foreach my $browser (@{$self->{'browser'}})
   {
      my $name = $browser->{'name'}->[0];
      $self->{'dbh'}->do ("delete from $self->{'dbindex'}$name where identifier=\'$identifier\'");
   }
}


# add a new XML record to the index database
sub indexXML
{
   my ($self, $xml, $identifier) = @_;

   my $mainrow = '';
      
   # iterate over the list of browser definitions
   foreach my $browser (@{$self->{'browser'}})
   {
      # get baseline browser data
      my $name = $browser->{'name'}->[0];
#      my $field = $browser->{'field'}->[0];
      my $type = $browser->{'type'}->[0];
      
      # get list of all eligible fields in metadata
      my @datafields = ();
      foreach my $field (@{$browser->{'field'}})
      {
         my @fieldcomponents = split ('/', $field);
         my @datafields2 = ( $xml );
         foreach my $subfield (@fieldcomponents)
         {
            my @newdatafields = ();
            foreach my $datafield (@datafields2)
            {
               foreach my $newcandidate ($datafield->getElementsByTagName ($subfield, 0))
               {
                  push (@newdatafields, $newcandidate);
               }
            }
            @datafields2 = @newdatafields;
         }
         @datafields = (@datafields, @datafields2);
      }
      
      # escape for sql queries
      $identifier =~ s/'/\\\'/g;

      # iterate over all eligible fields
      my $firsttime = 1;
      my %cache = ();
      foreach my $datafield (@datafields)
      {
         my $data = $datafield->getChildNodes->toString;
            
         # do regular expression transformations
         foreach my $retransform (@{$browser->{'retransform'}})
         {
            my $from = $retransform->{'from'}->[0];
            my $to = $retransform->{'to'}->[0];
            if ($data =~ $from)
            {
               $data =~ s/$from/$to/ee;
               last;
            }
         }
               
         # escape for sql queries
         $data       =~ s/'/\\\'/g;
               
         # split up into components
         my @dataparts = ($data);
         if ((exists $browser->{'separator'}) && ($#{$browser->{'separator'}} == 0) &&
             ($browser->{'separator'}->[0] ne ''))
         {
            @dataparts = split ($browser->{'separator'}->[0], $data);
         }
         
         # add entries to browse table
         my $prefix = '';
         foreach my $datapart (@dataparts)
         {
            if (! exists $cache{$prefix.$datapart})
            {
               $self->{'dbh'}->do ("insert into $self->{'dbindex'}$name values (\'$identifier\', \'$prefix$datapart\')");
               $cache{$prefix.$datapart} = 1;
            }
            if ($#dataparts > 0)
            {
               $prefix .= $datapart.$browser->{'separator'}->[0];
            }
         }
               
         # add to main row entry
         if ($firsttime == 1)
         {
            $mainrow .= ",\'".$data."\'"; 
            $firsttime = 0; 
         }
      }
      
      # add blank field if not found
      if ($firsttime == 1)
      {
         $mainrow .= ",\'\'";
      }
   }

   # add entry to main browse table
   $self->{'dbh'}->do ("insert into $self->{'dbindex'} values (\'$identifier\'$mainrow)");
}


1;


