#  ---------------------------------------------------------------------
#   OAI Data Provider template (OAI-PMH v1.1)
#    v2.03
#    April 2002
#  ------------------+--------------------+-----------------------------
#   Hussein Suleman  |   hussein@vt.edu   |    www.husseinsspace.com    
#  ------------------+--------------------+-+---------------------------
#   Department of Computer Science          |        www.cs.vt.edu       
#     Digital Library Research Laboratory   |       www.dlib.vt.edu      
#  -----------------------------------------+-------------+-------------
#   Virginia Polytechnic Institute and State University   |  www.vt.edu  
#  -------------------------------------------------------+-------------


package OAI::OAIDP;


use Pure::EZCGI;
use OAI::Utility;


# constructor
sub new
{
   my ($classname) = @_;

   my $self = {
      class           => $classname,
      xmlnsprefix     => 'http://www.openarchives.org/OAI/1.1',
      protocolversion => '1.1',
      repositoryName  => 'NoName Repository',
      adminEmail      => 'mailto:someone@somewhere',
      metadatanamespace => {
         oai_dc       => 'http://purl.org/dc/elements/1.1/',
      },
      metadataschema => {
         oai_dc       => 'http://www.openarchives.org/OAI/dc.xsd',
      },
      metadataroot => {
         oai_dc       => 'dc',
      },
      metadatarootparameters => {
         oai_dc       => '',
      },
      utility         => new OAI::Utility,
   };

   bless $self, $classname;
   return $self;
}


# destructor
sub dispose
{
   my ($self) = @_;
}


# output XML HTTP header
sub xmlheader
{
   my ($self) = @_;

   # calculate timezone automatically
   my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime (time);
   my $timezone = sprintf ("%+03d:00", ((((($hour-(gmtime)[2]) % 24) + 11) % 24) - 11));
   my $datestring = sprintf ("%04d-%02d-%02dT%02d:%02d:%02d%s",
                    $year+1900, $mon+1, $mday, $hour, $min, $sec,
                    $timezone);

   "Content-type: text/xml\n\n".
   "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n\n".
   "<".$self->{'cgi'}->param ('verb')." xmlns=\"$self->{'xmlnsprefix'}/OAI_".$self->{'cgi'}->param ('verb')."\" ".
   "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" ".
   "xsi:schemaLocation=\"$self->{'xmlnsprefix'}/OAI_".$self->{'cgi'}->param ('verb')." ".
   "$self->{'xmlnsprefix'}/OAI_".$self->{'cgi'}->param ('verb').".xsd\">\n\n".
   "<responseDate>$datestring</responseDate>\n".
   "<requestURL>".
   $self->{'utility'}->lclean ($self->{'cgi'}->{'requestURL'}).
   "</requestURL>\n\n";
}


# output XML HTTP footer
sub xmlfooter
{
   my ($self) = @_;
   "\n</".$self->{'cgi'}->param ('verb').">\n";
}


# generate an HTTP error message
sub Error
{
   my ($self, $errmsg, $errno) = @_;
   "Status: $errno $errmsg\n\n";
}


# check for the validity of the date according to the OAI spec
sub DateisValid
{
   my ($self, $date) = @_;

   if ($date !~ /^[0-9]{4}-[0-9]{2}-[0-9]{2}$/)
   { return 0; }
   
   my ($year, $month, $day, $daysinmonth);
   $year = substr ($date, 0, 4);
   $month = substr ($date, 5, 2);
   $day = substr ($date, 8, 2);

   if ((((($year % 4) == 0) && (($year % 100) != 0)) || (($year % 400) == 0))
       && ($month == 2))
   { $daysinmonth = 29; }
   elsif (($month == 4) || ($month == 6) || ($month == 9) || ($month == 11))
   { $daysinmonth = 30; }
   elsif ($month == 2)
   { $daysinmonth = 28; }
   else
   { $daysinmonth = 31; }

   if (($day == 0) || ($day > $daysinmonth))
   { return 0; }
   if (($month == 0) || ($month > 12))
   { return 0; }
   if ($year == 0)
   { return 0; }

   return 1;
}


# check of the metadata format is valid
sub MetadataFormatisValid
{
   my ($self, $metadataFormat) = @_;

   my $found = 0;
   foreach my $i (keys %{$self->{metadatanamespace}})
   {
      if ($metadataFormat eq $i)
      { $found = 1; }
   }

   if ($found == 1)
   { return 1; }
   else
   { return 0; }
}


# format the record by encapsulating it in a "record" container
sub FormatRecord
{
   my ($self, $identifier, $datestamp, $metadata, $about, $status) = @_;
   
   my $statusattribute = '';
   if ((defined $status) && ($status eq 'deleted'))
   {
      $statusattribute = " status=\"deleted\"";
   }

   my $output =
      "<record$statusattribute>\n".
      "<header>\n".
      "<identifier>$identifier</identifier>\n".
      "<datestamp>$datestamp</datestamp>\n".
      "</header>\n";
   
   if ((defined $metadata) && ($metadata ne ''))
   {
      $output .= "<metadata>\n$metadata</metadata>\n";
   }
   if ((defined $about) && ($about ne ''))
   {
      $output .= "<about>\n$about</about>\n";
   }
                                 
   $output."</record>\n";
}


# standard handler for Identify verb
sub Identify
{
   my ($self) = @_;

   my $identity = $self->Archive_Identify;
   if (! exists $identity->{'repositoryName'})
   {
      $identity->{'repositoryName'} = $self->{'repositoryName'};
   }
   if (! exists $identity->{'adminEmail'})
   {
      $identity->{'adminEmail'} = $self->{'adminEmail'};
   }
   $identity->{'protocolVersion'} = $self->{'protocolversion'};
   $identity->{'baseURL'} = $self->{'cgi'}->{'baseURL'};
   $identity->{'mdorder'} = [ qw ( repositoryName baseURL protocolVersion adminEmail description ) ];

   $self->xmlheader.
   $self->{'utility'}->FormatXML ($identity).
   $self->xmlfooter;
}


# standard handler for ListMetadataFormats verb
sub ListMetadataFormats
{
   my ($self) = @_;
   
   my $identifier = $self->{'cgi'}->param ('identifier');
   my $metadataNamespace = $self->{'metadatanamespace'};
   my $metadataSchema = $self->{'metadataschema'};

   my $lmf = $self->Archive_ListMetadataFormats ($identifier);
   if ($#$lmf > 0)
   {
      $metadataNamespace = $$lmf[0];
      $metadataSchema = $$lmf[1];
   }

   my $buffer = $self->xmlheader;
   my $i;
   foreach $i (keys %{$metadataNamespace})
   {
      $buffer .= "<metadataFormat>\n".
                 "<metadataPrefix>$i</metadataPrefix>\n".
                 "<schema>$metadataSchema->{$i}</schema>\n".
                 "<metadataNamespace>$metadataNamespace->{$i}</metadataNamespace>\n".
                 "</metadataFormat>\n";
   }
   $buffer.$self->xmlfooter;
}


# standard handler for ListSets verb
sub ListSets
{
   my ($self) = @_;

   my $buffer = $self->xmlheader;
   
   my $setlist = $self->Archive_ListSets;
   my $item;
   
   foreach $item (@$setlist)
   {
      $buffer .= "<set>\n".
                 "  <setSpec>".$self->{'utility'}->lclean ($$item[0])."</setSpec>\n".
                 "  <setName>".$self->{'utility'}->lclean ($$item[1])."</setName>\n".
                 "</set>\n";
   }
   $buffer.$self->xmlfooter;
}


# standard handler for GetRecord verb
sub GetRecord
{
   my ($self) = @_;

   my ($identifier) = $self->{'cgi'}->param ('identifier');
   my ($metadataPrefix) = $self->{'cgi'}->param ('metadataPrefix');

   if ($metadataPrefix eq '')
   {
      return $self->Error ("missing metadataPrefix parameter", 400);
   }
   if ($identifier eq '')
   {
      return $self->Error ("missing identifier parameter", 400);
   }

   my $recref = $self->Archive_GetRecord ($identifier, $metadataPrefix);

   my $buffer = $self->xmlheader;
   if ($recref)
   {
      $buffer .= $self->Archive_FormatRecord ($recref, $metadataPrefix);
   }
   $buffer.$self->xmlfooter;
}


# standard handler for ListRecords verb
sub ListRecords
{
   my ($self) = @_;

   my ($set, $from, $until, $metadataPrefix);
   my ($resumptionToken, $allrows);

   $resumptionToken = $self->{'cgi'}->param ('resumptionToken');
   if ($resumptionToken eq '')
   {
      $set = $self->{'cgi'}->param ('set');
      $from = $self->{'cgi'}->param ('from');
      $until = $self->{'cgi'}->param ('until');
      $metadataPrefix = $self->{'cgi'}->param ('metadataPrefix');

      if ($metadataPrefix eq '')
      {
         return $self->Error ("missing metadataPrefix parameter", 400);
      }
      if (($from ne '') && (!($self->DateisValid ($from))))
      {
         return $self->Error ("illegal from parameter", 400);
      }
      if (($until ne '') && (!($self->DateisValid ($until))))
      {
         return $self->Error ("illegal until parameter", 400);
      }
   }

   ($allrows, $resumptionToken, $metadataPrefix) =  
     $self->Archive_ListRecords ($set, $from, $until, $metadataPrefix, $resumptionToken);

   my $buffer = $self->xmlheader;
   foreach my $recref (@$allrows)
   { 
      $buffer .= $self->Archive_FormatRecord ($recref, $metadataPrefix);
   }
   if ($resumptionToken ne '')
   {
      $buffer .= "<resumptionToken>$resumptionToken</resumptionToken>\n";
   }
   $buffer.$self->xmlfooter;
}


# standard handler for ListIdentifiers verb
sub ListIdentifiers
{
   my ($self) = @_;

   my ($set, $from, $until);
   my ($resumptionToken, $allrows);

   $resumptionToken = $self->{'cgi'}->param ('resumptionToken');
   if ($resumptionToken eq '')
   {
      $set = $self->{'cgi'}->param ('set');
      $from = $self->{'cgi'}->param ('from');
      $until = $self->{'cgi'}->param ('until');

      if (($from ne '') && (!($self->DateisValid ($from))))
      {
         return $self->Error ("illegal from parameter", 400);
      }
      if (($until ne '') && (!($self->DateisValid ($until))))
      {
         return $self->Error ("illegal until parameter", 400);
      }
   }

   ($allrows, $resumptionToken) = 
     $self->Archive_ListIdentifiers ($set, $from, $until, $resumptionToken);

   my $buffer = $self->xmlheader;
   foreach my $recref (@$allrows)
   {
      my $status = '';
      if ((exists $recref->{'status'}) && ($recref->{'status'} eq 'deleted'))
      { $status = " status=\"deleted\""; }
      $buffer .= "<identifier$status>$recref->{'urn'}</identifier>"
   }
   if ($resumptionToken ne '')
   {
      $buffer .= "<resumptionToken>$resumptionToken</resumptionToken>\n";
   }
   $buffer.$self->xmlfooter;
}


# stub routines to get actual data from archives


sub Archive_FormatRecord
{
   my ($self, $recref, $metadataFormat) = @_;
   
   $self->FormatRecord ('identifier',
                        '1000-01-01',
                        $self->{'utility'}->FormatXML ({}),
                        $self->{'utility'}->FormatXML ({}),
                        ''
                       );
}


sub Archive_Identify
{
   my ($self) = @_;

   {};
}


sub Archive_ListSets
{
   my ($self) = @_;
   
   [];
}


sub Archive_ListMetadataFormats
{
   my ($self, $identifier) = @_;
   
   [];
}


sub Archive_GetRecord
{
   my ($self, $identifier, $metadataPrefix) = @_;
   
   my %records = ();

   undef;
}


sub Archive_ListRecords
{
   my ($self, $set, $from, $until, $metadataPrefix, $resumptionToken) = @_;
   
   my $results = [];
   my @allrows = ();
   $resumptionToken = '';

   ( \@allrows, $resumptionToken, $metadataPrefix );
}


sub Archive_ListIdentifiers
{
   my ($self, $set, $from, $until, $resumptionToken) = @_;
   
   my $results = [];
   my @allrows = ();
   $resumptionToken = '';

   ( \@allrows, $resumptionToken );
}


# main loop to process parameters and call appropriate verb handler
sub Run
{
   my ($self) = @_;

   if (! exists $self->{'cgi'})
   {
      $self->{'cgi'} = new Pure::EZCGI;
   }

   # run appropriate handler procedure
   if ($self->{'cgi'}->param ('verb') eq 'Identify')
   { print $self->Identify; }
   elsif ($self->{'cgi'}->param ('verb') eq 'ListMetadataFormats')
   { print $self->ListMetadataFormats; }
   elsif ($self->{'cgi'}->param ('verb') eq 'GetRecord')
   { print $self->GetRecord; }
   elsif ($self->{'cgi'}->param ('verb') eq 'ListSets')
   { print $self->ListSets; }
   elsif ($self->{'cgi'}->param ('verb') eq 'ListRecords')
   { print $self->ListRecords; }
   elsif ($self->{'cgi'}->param ('verb') eq 'ListIdentifiers')
   { print $self->ListIdentifiers; }
   else
   { print $self->Error ('unsupported verb', 400); }
}


1;


# HISTORY
#
# 2.01
#  fixed ($identifier) error
#  added status to FormatRecord
# 2.02
#  added metadataPrefix to GetRecord hander
# 2.03
#  fixed errors in xmlheader
