#!/usr/bin/perl -I. -w
use strict;

#  ---------------------------------------------------------------------
#   configuration script for DBBrowse v1.0
#    v1.0
#    April 2002
#  ------------------+--------------------+-----------------------------
#   Hussein Suleman  |   hussein@vt.edu   |    www.husseinsspace.com    
#  ------------------+--------------------+-+---------------------------
#   Department of Computer Science          |        www.cs.vt.edu       
#     Digital Library Research Laboratory   |       www.dlib.vt.edu      
#  -----------------------------------------+-------------+-------------
#   Virginia Polytechnic Institute and State University   |  www.vt.edu  
#  -------------------------------------------------------+-------------

# configure the OAI data provider layer


use lib '../lib/';

use Pure::EZXML;
use Pure::EZHTTP;
use Pure::X2D;


# get data through HTTP, with retries
sub getHTTP
{
   my ($url) = @_;

   # create a user agent object
   my $ua = new Pure::EZHTTP;
   $ua->agent("VT OAI Harvester/1.0 " . $ua->agent);

   # create a request
   my $req = new Pure::EZHTTP GET => $url;

   my $state = 0;
   my $res;
   while ($state == 0)
   {
      # pass request to the user agent and get a response back
      $res = $ua->request($req);
      
      if ($res->code == 503)
      {
         my $sleep = $res->header ('Retry-After');
         if (not defined ($sleep) || ($sleep < 0) || ($sleep > 86400))
         { $state = 1;}
         else
         { sleep ($sleep); }
      }
      else
      { $state = 1; }
   }
   
   # return results to caller
   $res;
}


# read in configuration and set best-guess defaults for missing values
sub ReadConfig
{
   my ($configname) = @_;
   
   my $configfile = "$configname/config.xml";
   my $config = new Pure::X2D ($configfile);
   
   if ((! defined $config->{'repositoryName'}) || ($config->{'repositoryName'}->[0] eq ''))
   {
      $config->{repositoryName} = [ 'ODL Browse Engine' ];
   }

   if ((! defined $config->{'adminEmail'}) || ($config->{'adminEmail'}->[0] eq ''))
   {
      $config->{'adminEmail'} = [ '' ];

      # guess admin id from hostname and user logged in
      my $hostname = `hostname`; 
      my $username = `whoami`;
      if ((defined $hostname) && ($hostname ne '') && 
          (defined $username) && ($username ne ''))
      {
         $hostname =~ s/[\r\n\t\s]//go;
         $username =~ s/[\n\r\t\s]//go;
         
         $config->{'adminEmail'} = [ $username.'@'.$hostname ];
      }
   }

   if ((! defined $config->{'database'}) || ($config->{'database'}->[0] eq ''))
   { $config->{'database'} = [ 'odlbrowse' ]; }
   if ((! defined $config->{'dbusername'}) || ($config->{'dbusername'}->[0] eq ''))
   {
      $config->{'dbusername'} = [ '' ];
      
      my $username = `whoami`;
      if ((defined $username) && ($username ne ''))
      {
         $username =~ s/[\r\t\n\s]//go;
         $config->{'dbusername'} = [ $username ]; 
      }
   }
   if ((! defined $config->{'dbpassword'}) || ($config->{'dbpassword'}->[0] eq ''))
   { $config->{'dbpassword'} = [ '' ]; }
   if ((! defined $config->{'table'}) || ($config->{'table'}->[0] eq ''))
   { $config->{'table'} = [ $configname ]; }
   if ((! defined $config->{'recordlimit'}) || ($config->{'recordlimit'}->[0] eq ''))
   { $config->{'recordlimit'} = [ '' ]; }
   
   $config->{'mdorder'} = [ qw ( repositoryName adminEmail database dbusername dbpassword table recordlimit archive browser ) ];

   if (! defined $config->{'archive'})
   { $config->{'archive'} = []; }

   foreach my $archive (@{$config->{'archive'}})
   {
      if ((! defined $archive->{'identifier'}) || ($archive->{'identifier'} eq ''))
      { $archive->{'identifier'} = [ $configname ]; }
      if ((! defined $archive->{'url'}) || ($archive->{'url'} eq ''))
      { $archive->{'url'} = [ '' ]; }
      if ((! defined $archive->{'metadataPrefix'}) || ($archive->{'metadataPrefix'} eq ''))
      { $archive->{'metadataPrefix'} = [ 'oai_dc' ]; }
      if ((! defined $archive->{'interval'}) || ($archive->{'interval'} eq ''))
      { $archive->{'interval'} = [ 86400 ]; }
      if ((! defined $archive->{'interrequestgap'}) || ($archive->{'interrequestgap'} eq ''))
      { $archive->{'interrequestgap'} = [ 10 ]; }
      if ((! defined $archive->{'set'}) || ($archive->{'set'} eq ''))
      { $archive->{'set'} = [ '' ]; }
      if ((! defined $archive->{'overlap'}) || ($archive->{'overlap'} eq ''))
      { $archive->{'overlap'} = [ 0 ]; }
      if ((! defined $archive->{'granularity'}) || ($archive->{'granularity'} eq ''))
      { $archive->{'granularity'} = [ '' ]; }

      $archive->{'mdorder'} = [ qw ( identifier url metadataPrefix interval interrequestgap set overlap granularity ) ];
   }

   if (! defined $config->{'browser'})
   { $config->{'browser'} = []; }

   foreach my $browser (@{$config->{'browser'}})
   {
      if ((! defined $browser->{'name'}) || ($browser->{'name'} eq ''))
      { $browser->{'name'} = [ '' ]; }
      if ((! defined $browser->{'field'}) || ($browser->{'field'} eq ''))
      { $browser->{'field'} = [ '' ]; }
      if ((! defined $browser->{'type'}) || ($browser->{'type'} eq ''))
      { $browser->{'type'} = [ 'freetext' ]; }
      if ((! defined $browser->{'separator'}) || ($browser->{'separator'} eq ''))
      { $browser->{'separator'} = [ '' ]; }
      $browser->{'mdorder'} = [ qw ( name field type separator retransform ) ];

      if (! defined $browser->{'retransform'})
      { $browser->{'retransform'} =  []; }
      
      foreach my $retransform (@{$browser->{'retransform'}})
      {
         if ((! defined $retransform->{'from'}) || ($retransform->{'from'} eq ''))
         { $retransform->{'from'} = [ '' ]; }
         if ((! defined $retransform->{'to'}) || ($retransform->{'to'} eq ''))
         { $retransform->{'to'} = [ '' ]; }

         $retransform->{'mdorder'} = [ qw ( from to ) ];
      }
   }

   $config;
}


# write configuration data and template files
sub WriteConfig 
{
   my ($config, $configname) = @_;
   
   mkdir ($configname);
   
   open (CONFIG, ">$configname/config.xml");
   print CONFIG "<?xml version=\"1.0\" ?>\n\n";
   print CONFIG "<odlbrowse xmlns=\"http://oai.dlib.vt.edu/ODL/DBBrowse/config\"\n".
                "    xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n".
                "    xsi:schemaLocation=\"http://oai.dlib.vt.edu/ODL/DBBrowse/config\n".
                "                         http://oai.dlib.vt.edu/ODL/DBBrowse/config.xsd\"\n".
                ">\n\n";
   print CONFIG $config->toXML;
   print CONFIG "</odlbrowse>\n";
   close (CONFIG);
   
   system ("cp template/* $configname/");
   mkdir ("$configname/data");
}


my $fastforward = 0;

# input a line of data, with prompt and default value
sub InputLine
{
   my ($variable, $prompt) = @_;
   
   if (defined $prompt)
   {
      print "$prompt [$$variable] : ";
   }
   if ($fastforward == 1)
   {
      print "$$variable\n";
      select(undef, undef, undef, 0.25);
   }
   else
   {
      my $line = <STDIN>;
      chomp $line;
      if ($line eq '&continue')
      {
         $fastforward = 1;
         print "$$variable\n";
      }
      elsif ($line eq '&delete')
      {
         $$variable = '';
         InputLine ($variable, $prompt);
      }
      elsif ($line ne '')
      {
         $$variable = $line;
      }
   }
}


# wait for user to press ENTER
sub InputEnter
{
   my $temp = '';
   InputLine (\$temp);
}


# check validity of input and print error if necessary
sub InvalidInput
{
   my ($check, $errormsg) = @_;
   
   if ($check)
   {
      print "$errormsg\n";
      $fastforward = 0;
      return 1;
   }
   0;
}


# check for database operations on a single table
sub CheckTable
{
   my ($dbh, $table) = @_;

   my $sth = $dbh->prepare ("select count(*) from $table");
   if (! defined $sth)
   {
      return "Error preparing a select for table $table";
   }
   if (!($sth->execute))
   {
      $sth->finish;
      return "Error executing a select on table $table";
   }
   if (!($sth->fetchrow_hashref))
   {
      $sth->finish;
      return "Error getting row from table $table";
   }
   $sth->finish;
   return '';
}


# get and display a sample record
sub getsamplerecord
{
   my ($url, $metadataPrefix, $set) = @_;
   
   if ($set ne '')
   {
      $set = '&set='.$set;
   }

   # submit ListIdentifiers to archive
   my $res = getHTTP ($url.'?verb=ListIdentifiers'.$set);
   if ($res->code != 200)
   {
      print "Error: Archive does not respond to ListIdentifiers!\n";
      return;
   }

   # parse response and get first identifier   
   my $parser = new Pure::EZXML;
   my $lidoc = $parser->parse ($res->content)->getDocumentElement;
   my $identifiers = $lidoc->getElementsByTagName ('identifier', 0);
   if ($identifiers->getLength == 0)
   {
      print "Error: Cannot find an identifier!\n";
      return;
   }
   my $idnumber = int (rand ($identifiers->getLength));
   my $identifier = $identifiers->item($idnumber)->getChildNodes->toString;
   
   # submit GetRecord to archive
   my $res2 = getHTTP ($url.'?verb=GetRecord&metadataPrefix='.$metadataPrefix.'&identifier='.$identifier);
   if ($res->code != 200)
   {
      print "Error: Archive does not respond to GetRecord!\n";
      return;
   }
   
   # parse and search for record within response
   my $parser2 = new Pure::EZXML;
   my $grdoc = $parser2->parse ($res2->content)->getDocumentElement;
   my $records = $grdoc->getElementsByTagName ('record', 0);
   if ($records->getLength == 0)
   {
      print "Error: Cannot find record!\n";
      return;
   }
   my $metadata = $records->item(0)->getElementsByTagName ('metadata', 0);
   if ($metadata->getLength == 0)
   {
      print "Error: Cannot find metadata container!\n";
      return;
   }

   # output record to terminal
   print "Sample metadata record:\n";
   print $metadata->item(0)->getChildNodes->toString;
   print "\n\n";
}

# transform configuration for a single transformation
sub retransform_config
{
   my ($retransform) = @_;

   # FROM
   print "\n[TRANSFORM FROM]\n";
   print "Enter the (Perl) regular expression to match on\n";
   print "Example: Computer(.*)\n\n";
   do {
      InputLine (\$retransform->{'from'}->[0], "FROM");
   } while (InvalidInput (($retransform->{'from'}->[0] eq ''),
                          "This field cannot be left blank. Please enter a value."));


   # TO
   print "\n[TRANSFORM TO]\n";
   print "Enter the (Perl) regular expression to use as a replacement\n";
   print "Example: Computer\n\n";
   do {
      InputLine (\$retransform->{'to'}->[0], "TO");
   } while (InvalidInput (($retransform->{'to'}->[0] eq ''),
                          "This field cannot be left blank. Please enter a value."));
}


# browser configuration for a single browsable field
sub browser_config
{
   my ($browser) = @_;

   # Browser name
   print "\n[BROWSABLE FIELD NAME]\n";
   print "You need a unique name by which to refer to the field in\n".
         "queries\n";
   print "Examples: title, author\n\n";
   do {
      InputLine (\$browser->{'name'}->[0], "Field name");
   } while (InvalidInput (($browser->{'name'}->[0] eq ''),
                          "This field cannot be left blank. Please enter a value."));


   # Browser field
   print "\n[BROWSABLE FIELD SPECIFICATION]\n";
   print "What is the path from the root XML node to the node\n".
         "containing the data to be indexed ?\n";
   print "Examples: dc/title, etdms/thesis/grantor\n\n";
   do {
      InputLine (\$browser->{'field'}->[0], "Field specification");
   } while (InvalidInput (($browser->{'field'}->[0] eq ''),
                          "This field cannot be left blank. Please enter a value."));


   # Browser type
   print "\n[BROWSABLE FIELD TYPE]\n";
   print "The browsable field may be either \"freetext\" or \"controlled\"\n".
         "\"freetext\" means that the vocabulary is not constrained\n".
         "  and the engine will only sort on this field\n".
         "\"controlled\" means that the vocabulary is controlled\n".
         "  and the engine can restrict results based on this field\n\n";
   do {
      InputLine (\$browser->{'type'}->[0], "Field type");
   } while (InvalidInput ((($browser->{'type'}->[0] ne 'freetext') && ($browser->{'type'}->[0] ne 'controlled')),
                          "This field has to be either freetext of controlled. Please choose one."));


   # Browser separator
   if ($browser->{'type'}->[0] eq 'controlled')
   {
      print "\n[BROWSABLE FIELD SEPARATOR]\n";
      print "To support hierarchical information within the field, you need\n".
            "to specify the character or character sequence that separates\n".
            "components of the hierarchy. Leave blank for no hierarchies.\n".
            "Examples: / (e.g., Hardware/Input/Mouse)\n".
            "            (e.g., Hardware Input Mouse)\n".
            "          -- (e.g., Hardware--Input--Mouse)\n\n";
      InputLine (\$browser->{'separator'}->[0], "Field component separator");
   }


   # Transform configuration
   my $choice;
   print "\n[BROWSABLE FIELD TRANSFORMATIONS]\n";
   print "Each source field may be transformed by a Perl regular expression before\n".
         "classification. The first matching expression is applied and the rest are\n".
         "ignored. These transformations are used in a perl s/// expression so\n".
         "anything valid in such expression is valid here.\n";
   print "Examples: from=^([^ ]+) Engineering\$ to=\$1\n\n";
   do {
      # print out transform listing
      print "\nCurrent list of transformations:\n";
      if ($#{$browser->{'retransform'}} == -1)
      {
         print "No transformations currently defined !\n";
      }
      for ( my $i=0; $i<=$#{$browser->{'retransform'}}; $i++ )
      {
         my $retransform = $browser->{'retransform'}->[$i];
         print "$i. FROM=".$retransform->{'from'}->[0].' TO='.$retransform->{'to'}->[0]."\n";
      }
      
      # print out menu
      my $instrline = "\nSelect from: [A]dd   [R]emove   [E]dit   [D]one   Move [U]p   Move Dow[N]\n";
      if ($#{$browser->{'retransform'}} == -1)
      {
         $instrline = "\nSelect from: [A]dd   [D]one\n";
      }
      print $instrline;

      # get choice
      do {
         $choice = 'D';
         InputLine (\$choice, "Enter your choice");
         $choice = uc ($choice);
      } while (InvalidInput (((($choice !~ /^[AREDUN]$/) || ($#{$browser->{'retransform'}} == -1)) &&
                              (($choice !~ /^[AD]$/) || ($#{$browser->{'retransform'}} > -1))),
                             $instrline));

      # act on choice
      if ($choice eq 'A')
      {
         push (@{$browser->{'retransform'}}, { from => [''], to => [''], 
           mdorder => [ qw ( from to ) ] } );
         retransform_config ($browser->{'retransform'}->[-1]);
      }
      elsif (($choice eq 'E') || ($choice eq 'R') || ($choice eq 'U') || ($choice eq 'N'))
      {
         my $retransformno = '';
         do {
            InputLine (\$retransformno, "Enter the number of the field");
         } while (InvalidInput ((($retransformno !~ /^[0-9]+$/) || ($retransformno > $#{$browser->{'retransform'}})),
                                "This has to be an integer in the range listed. Please try again."));
         
         if ($choice eq 'R')
         {
            my $confirm = 'Y/N';
            do {
               InputLine (\$confirm, "Confirm");
               $confirm = uc ($confirm);
            } while (InvalidInput (($confirm !~ /^[NY]$/),
                                   "Enter Y or N"));
            if ($confirm eq 'Y')
            {
               splice (@{$browser->{'retransform'}}, $retransformno, 1);
            }
         }
         elsif ($choice eq 'E')
         {
            retransform_config ($browser->{'retransform'}->[$retransformno]);
         }
         elsif ($choice eq 'U')
         {
            if ($retransformno > 0)
            {
               ($browser->{'retransform'}->[$retransformno], $browser->{'retransform'}->[$retransformno-1]) =
                ($browser->{'retransform'}->[$retransformno-1], $browser->{'retransform'}->[$retransformno])
            }   
         }
         elsif ($choice eq 'N')
         {
            if ($retransformno < $#{$browser->{'retransform'}})
            {
               ($browser->{'retransform'}->[$retransformno], $browser->{'retransform'}->[$retransformno+1]) =
                ($browser->{'retransform'}->[$retransformno+1], $browser->{'retransform'}->[$retransformno])
            }
         }
      }
      
   } while ($choice ne 'D');

}


# archive configuration for a single archive
sub archive_config
{
   my ($archive) = @_;

   # Archive Identifier
   print "\n[ARCHIVE IDENTIFIER]\n";
   print "You need a unique name by which to refer to the archive you\n".
         "will harvest metadata from\n";
   print "Examples: $ARGV[0], VTETD\n\n";
   do {
      InputLine (\$archive->{'identifier'}->[0], "Archive identifier");
   } while (InvalidInput (($archive->{'identifier'}->[0] eq ''),
                          "This field cannot be left blank. Please enter a value."));


   # Archive URL
   my $ok = 0;
   my $gotodl = 0;
   print "\n[ARCHIVE URL]\n";
   print "This is the baseURL of the archive being harvested from.\n\n";
   do {
      do {
         InputLine (\$archive->{'url'}->[0], "Archive baseURL");
      } while (InvalidInput (($archive->{'url'}->[0] eq ''),
                             "This field cannot be left blank. Please enter a value."));

      print "Issuing Identify request to archive ...\n";
      
      # submit Identify to archive
      my $res = getHTTP ($archive->{'url'}->[0].'?verb=Identify');
      print "Got response code : ".$res->code."\n";
      
      if ($res->code == 200)
      {
         $ok = 1;
         
         my $parser = new Pure::EZXML;
         my $document = $parser->parse ($res->content)->getDocumentElement;
         
         foreach my $description ($document->getElementsByTagName ('description', 0))
         {
            foreach my $odldescription ($description->getElementsByTagName ('odl-description', 0))
            {
               $gotodl = 1;
            }
         }
         
         if ($archive->{'overlap'}->[0] == 0)
         {
            if ($gotodl == 1)
            {
               $archive->{'overlap'}->[0] = 86401;
            }
            else
            {
               $archive->{'overlap'}->[0] = 172800;
            }
         }
         
         if ($archive->{'granularity'}->[0] eq '')
         {
            if ($gotodl == 1)
            {
               $archive->{'granularity'}->[0] = 'second';
            }
            else
            {
               $archive->{'granularity'}->[0] = 'day';
            }
         }
      }
      else
      {
         print "Error with archive or URL - please check\n";
      }
   } while ($ok == 0);


   # harvesting interval
   print "\n[HARVESTING INTERVAL]\n";
   print "Choose how often you want to harvest (in seconds)\n".
         "86400 = daily, 3600 = hourly\n\n";
   do {
      InputLine (\$archive->{interval}->[0], "Harvesting interval");
   } while (InvalidInput (($archive->{interval}->[0] < 1),
                          "This field must be number and >0. Please try again."));


   # seconds overlap
   print "\n[HARVESTING OVERLAP]\n";
   print "You need to select the number of seconds by which to overlap\n".
         "harvesting to handle differences in times\n";
   if ($gotodl == 1)
   {
      print "It seems you are harvesting from an ODL archive !\n".
            "Use 1 for an archive in the same timezone or\n".
            "86401 for an archive in a different or unknown timezone\n\n";
   }
   else
   {
      print "It seems you are harvesting from a standard OAI archive !\n".
            "Use 86400 for an archive in the same timezone or\n".
            "172800 for an archive in a different or unknown timezone\n\n";
   }
   do {
      InputLine (\$archive->{'overlap'}->[0], "Harvesting overlap");
   } while (InvalidInput (($archive->{'overlap'}->[0] < 1),
                          "This field must be a number and >0. Please try again."));


   # harvesting granularity
   print "\n[HARVESTING GRANULARITY]\n";
   print "This specifies whether harvesting is performed by day or second\n".
         "For OAI archives this is always day but for ODL archives this may\n".
         "be either\n\n";
   if ($gotodl == 1)
   {
      print "It seems you are harvesting from an ODL archive !\n".
            "Use \'day\' or \'second\'\n\n";
   }
   else
   {
      print "It seems you are harvesting from a standard OAI archive !\n".
            "Use \'day\'\n\n";
   }
   do {
      InputLine (\$archive->{'granularity'}->[0], "Harvesting granularity");
   } while (InvalidInput ((($archive->{'granularity'}->[0] ne 'day') && ($archive->{'granularity'}->[0] ne 'second')),
                          "This field must be either day or second. Please try again."));


   # metadataPrefix
   print "\n[metadataPrefix]\n";
   print "This is the metadata format that will be harvested and stored.\n\n";
   
   # submit ListMetadataFormats to archive and generate list of formats
   my $res = getHTTP ($archive->{'url'}->[0].'?verb=ListMetadataFormats');
   if ($res->code != 200)
   {
      print "Fatal error: Archive responds to Identify but not ListMetadataFormats!\n";
      exit;
   }
   my $parser = new Pure::EZXML;
   my $lmfdoc = $parser->parse ($res->content)->getDocumentElement;
   my @mdprefixes = ();
   foreach my $mdf ($lmfdoc->getElementsByTagName ('metadataFormat', 0))
   {
      foreach my $mdp ($mdf->getElementsByTagName ('metadataPrefix', 0))
      {
         push (@mdprefixes, $mdp->getChildNodes->toString);
      }
   }
   print "Valid formats: ".join (',', @mdprefixes);
   print "\n\n";
   do {
      InputLine (\$archive->{'metadataPrefix'}->[0], "metadataPrefix");
   } while (InvalidInput ((1 - grep {$_ eq $archive->{'metadataPrefix'}->[0]} @mdprefixes),
                          "Please choose a value from the list above."));


   # set
   print "\n[set]\n";
   print "Choose a set to harvested (descriptions are in brackets).\n".
         "Leave blank to harvest the whole archive.\n\n";
   
   # submit ListSets to archive and generate list of formats
   $res = getHTTP ($archive->{'url'}->[0].'?verb=ListSets');
   if ($res->code != 200)
   {
      print "Fatal error: Archive responds to Identify but not ListSets!\n";
      exit;
   }
   my $lsdoc = $parser->parse ($res->content)->getDocumentElement;
   my @sets = ();
   print "Valid sets:\n";
   foreach my $set ($lsdoc->getElementsByTagName ('set', 0))
   {
      my $setSpec = $set->getElementsByTagName ('setSpec', 0)->item(0)->getChildNodes->toString;
      my $setName = $set->getElementsByTagName ('setName', 0)->item(0)->getChildNodes->toString;
      push (@sets, $setSpec);
      print "  $setSpec [$setName]\n";
   }
   print "\n";
   do {
      InputLine (\$archive->{'set'}->[0], "set");
   } while (InvalidInput ((($archive->{'set'}->[0] ne '' ) && 
                           (1 - grep {$_ eq $archive->{'set'}->[0]} @sets)),
                          "Please choose a value from the list above."));
}


# main program body !
sub main 
{
   $| = 1;
   my $ok;
   
   print "+-------------------------------------------+\n".
         "| DBBrowse Categorical Browser Configurator |\n".
         "+-------------------------------------------+\n".
         "| Version 1.0 :: April 2002                 |\n".
         "| Hussein Suleman <hussein\@vt.edu>          |\n".
         "| Digital Library Research Laboratory       |\n".
         "| www.dlib.vt.edu :: Virginia Tech          |\n".
         "--------------------------------------------+\n\n";
         
   if (! exists $ARGV[0])
   {
      print "Missing configuration name\n\n";
      print "Syntax: configure.pl <configname>\n\n";
      exit;
   }
   
   my $config = ReadConfig ($ARGV[0]);
   
   print "Defaults/previous values are in brackets - press <enter> to accept those\n".
         "enter \"&delete\" to erase a default value\n".
         "enter \"&continue\" to skip further questions and use all defaults\n".
         "press <ctrl>-c to escape at any time (new values will be lost)\n".
         "\nPress <enter> to continue\n\n";
         
   InputEnter;

         
   # test for not finding DBI
   my $gotdbi = eval "require DBI";
   print "\n[DBI]\n";
   print "You need the DBI Perl module installed in order to use this script\n".
         "Checking ... ";
   if (! defined $gotdbi)
   {
      print "not installed !\n".
            "Please press CTRL-C and then install that module.\n".
            "You also need a module specific to your database e.g., DBD::mysql.\n".
            " \"man CPAN\" for help on how to do this.\n\n";
      exit;
   }
   else
   {
      print "ok\n\n";
   }
   

   # test for available driver
   my @dbi_drivers = DBI->available_drivers;
   my $driver_list = join (', ', @dbi_drivers);
   print "\n[DATABASE DRIVER]\n";
   print "DBI requires special drivers to access different databases. Please\n".
         "confirm that the list below contains the name of your database.\n".
         "If it is not here or you are not sure, press CTRL-C and install it\n";
   print "\n".$driver_list."\n\n";
   print "Press ENTER to continue\n";
   InputEnter;

   
   # Database Name + Username + Password
   $ok = 0;
   print "\n[DATABASE CONNECTION]\n";
   print "The DBBrowse scripts need to connect to a database to store\n".
         "and retrieve data. For this you need to supply the name of the\n".
         "database in the format required by your database\n".
         "(you will be asked for username and password next).\n".
         "If you dont have permission to access the database from this host\n".
         "you may want to install this OAI script on another machine or grant\n".
         "permissions before you continue. If the database does not exist, you\n".
         "probably should create it before continuing.\n";
   do {
      my @dbparts = ('DBI', 'mysql', 'odlbrowse');
      if ($config->{'database'}->[0] =~ /([^:]+):([^:]+):(.*)/)
      {
         @dbparts = ($1, $2, $3);
      }
   
      my $found = 0;
      do {
         print "\nDrivers: $driver_list\n";
         InputLine (\$dbparts[1], "Choose database driver from above list");
         foreach my $testdbidriver (@dbi_drivers)
         {
            if ($testdbidriver eq $dbparts[1])
            { $found = 1; }
         }
      } while (InvalidInput (($found == 0), "Invalid choice. Please choose again."));
      
      InputLine (\$dbparts[2], "Enter name of database");

      $dbparts[0] = 'DBI';
      $config->{'database'}->[0] = join (':', @dbparts);

      InputLine (\$config->{'dbusername'}->[0], "Enter the username to be used to connect");
      InputLine (\$config->{'dbpassword'}->[0], "Enter the password to be used to connect");

      print "Checking ... ";
      my $dbh = DBI->connect ($config->{'database'}->[0], $config->{'dbusername'}->[0], 
                              $config->{'dbpassword'}->[0], { PrintError=>0 });
      if (! defined $dbh)
      { 
         print "Cannot connect to the specified database with user/pass. Try again.\n";
         $fastforward = 0;
      }
      else
      {
         print "ok\n";
         $dbh->disconnect;
         $ok = 1;
      }
   } while ($ok == 0);
   

   # Database Table
   print "\n[DATABASE TABLE]\n";
   print "Within the database, multiple tables are needed for the indices\n".
         "Enter a unique table prefix for this component\n";
   print "Examples: dbbrowse, $ARGV[0]\n\n";
   do {
      InputLine (\$config->{'table'}->[0], "Table prefix");
   } while (InvalidInput (($config->{'table'}->[0] eq ''),
                          "This field cannot be left blank. Please enter a value."));


   # Repository Name
   print "\n[REPOSITORY NAME]\n";
   print "For identification purposes, you need to specify the full name of\n".
         "the component\n".
         "Example: JCDL ODL Browse Engine\n\n";
   do {
      InputLine (\$config->{'repositoryName'}->[0], "Enter your repository name");
   } while (InvalidInput (($config->{'repositoryName'}->[0] eq ''),
                          "This field cannot be left blank. Please enter a value."));
   

   # Admin Email
   print "\n[ADMINISTRATOR EMAIL]\n";
   print "When your browsing engine is used by others, any problems\n".
         "will be reported to an administator, whose email must be provided\n".
         "Examples: admin\@odlbrowse.org, provost\@university.edu.\n\n";
   do {
      InputLine (\$config->{'adminEmail'}->[0], "Enter your administrator's email");
   } while (InvalidInput (($config->{'adminEmail'}->[0] !~ /^[^@]+@[^@\.]+\.(.+)/), 
                          "That doesnt seem to be in the right format. Please try again."));


   # Archive administration
   if ($#{$config->{'archive'}} == -1)
   {
      push (@{$config->{'archive'}}, { identifier => [$ARGV[0]], url => [''], 
            metadataPrefix => [''], interval => ['86400'], interrequestgap => ['10'], 
            set => [''], overlap => ['0'], granularity => [''],
            mdorder => [ qw ( identifier url metadataPrefix interval interrequestgap set overlap granularity ) ] } );
   }
   archive_config ($config->{'archive'}->[-1]);
   

   # Browser configuration
   my $choice;
   print "\n[BROWSABLE FIELDS]\n";
   print "Add all the fields that should be indexed for browsing purposes\n";
   do {
      # print out archive listing
      print "\nCurrent list of browsable fields:\n";
      if ($#{$config->{'browser'}} == -1)
      {
         print "No fields currently defined !\n";
      }
      for ( my $i=0; $i<=$#{$config->{'browser'}}; $i++ )
      {
         my $browser = $config->{'browser'}->[$i];
         print "$i. ".$browser->{'name'}->[0].' '.$browser->{'field'}->[0].
               ' '.$browser->{'type'}->[0]." sep=\'".$browser->{'separator'}->[0].
               "\' [".($#{$browser->{'retransform'}}+1).' transforms]'.
               "\n";
      }
      
      # print out menu
      my $instrline = "\nSelect from: [A]dd   [R]emove   [E]dit   get [S]ample record   [D]one\n";
      if ($#{$config->{'browser'}} == -1)
      {
         $instrline = "\nSelect from: [A]dd   get [S]ample record   [D]one\n";
      }
      print $instrline;

      # get choice
      do {
         $choice = 'D';
         InputLine (\$choice, "Enter your choice");
         $choice = uc ($choice);
      } while (InvalidInput (((($choice !~ /^[SARED]$/) || ($#{$config->{'archive'}} == -1)) &&
                              (($choice !~ /^[SAD]$/) || ($#{$config->{'archive'}} > -1))),
                             $instrline));

      # act on choice
      if ($choice eq 'S')
      {
         my $archive = $config->{'archive'}->[-1];
         getsamplerecord ($archive->{'url'}->[0], $archive->{'metadataPrefix'}->[0], $archive->{'set'}->[0]);
      }
      elsif ($choice eq 'A')
      {
         push (@{$config->{'browser'}}, { name => [''], field => [''], 
           type => ['freetext'], separator => [''], retransform => [],
           mdorder => [ qw ( name field type separator retransform ) ] } );
         browser_config ($config->{'browser'}->[-1]);
      }
      elsif (($choice eq 'E') || ($choice eq 'R'))
      {
         my $browserno = '';
         do {
            InputLine (\$browserno, "Enter the number of the field");
         } while (InvalidInput ((($browserno !~ /^[0-9]+$/) || ($browserno > $#{$config->{'browser'}})),
                                "This has to be an integer in the range listed. Please try again."));
         
         if ($choice eq 'R')
         {
            my $confirm = 'Y/N';
            do {
               InputLine (\$confirm, "Confirm");
               $confirm = uc ($confirm);
            } while (InvalidInput (($confirm !~ /^[NY]$/),
                                   "Enter Y or N"));
            if ($confirm eq 'Y')
            {
               splice (@{$config->{'browser'}}, $browserno, 1);
            }
         }
         elsif ($choice eq 'E')
         {
            browser_config ($config->{'browser'}->[$browserno]);
         }
      }
      
   } while ($choice ne 'D');
   
   WriteConfig ($config, $ARGV[0]);
   
   print "\n\n\nFinis.\n\n";
   
   print "Now you are ready to harvest and test the browse engine\n".
         "Run $ARGV[0]/harvest.pl to try harvesting\n".
         "Use $ARGV[0]/browse.pl as the last bit of the baseURL for the engine interface\n\n".
         "NOTE: Remember to do a full harvest if you have modified any settings!\n\n";
}

main;
